#!/bin/sh

VERSION="2.0.22rc4-PKT202ready"

log() {
	local RED="\033[0;31m"
	local NONE="\033[0m"
	local LIGHT_PURPLE="\033[1;35m"
	local BLUE="\033[0;34m"
	if [ $# -eq 2 ]; then
		case $1 in
		warning)echo -e "${LIGHT_PURPLE}### $2${NONE}";;
		error)	echo -e "${RED}!!! $2${NONE}";;
		info)	echo -e "${BLUE}--> ${NONE}$2";;
		*)	echo -e "${NONE}$2";;
		esac
	else
		echo -e "${NONE}$1"
	fi
}

softusb_usage(){
#syntax softusb_usage()
	log echo "./softusb2.sh <command>
	commands:
		install <1st|2nd|ipbox> - install SoftUSB2
		uninstall - uninstall SoftUSB2
		set device <usb|ide> <fat|ext2> <label> [<ubootdev>] - set softusb configuration device
		set bootmenu <items> <timeout> - set bootmenu options
		set flash <auto|e2|pvrVER|dgs> [<dev>] - set flash image
		set image <1-9> <vfd-text> <auto|e2|pvrVER|dgs> softusb <file>
		set image <1-9> <vfd-text> <auto|e2|pvrVER|dgs> part <label> [<ubootdev>]
		set image <1-9> <vfd-text> <auto|e2|pvrVER|dgs> part <dev> <ubootdev>
		set image <1-9> <vfd-text> <auto|e2|pvrVER|dgs> partdir <dir> <label> [<ubootdev>]
		set image <1-9> <vfd-text> <auto|e2|pvrVER|dgs> partdir <dir> <dev> <ubootdev>
		set image <1-9> <vfd-text> <auto|e2|pvrVER|dgs> flash [<dev>]
		set image <1-9> <vfd-text> <auto|e2|pvrVER|dgs> nfs <path>
		set env <name> <value> - set custom uboot entries
		set net <ip> <netmask> <gateway> [<server>] - set network configuration
		set mac <mac> - set mac address
		set lastboot <num> [<yes|no>] - set last boot position, remember last selection or not
		mkimage <file> <name> [<size>] - make softusb image files from tar.gz
		mkswap [<size>] - make swap file
		extract <file> <dev/label> [<dir>] - extract archive to partition (and dir)
		status - print current configuration

Version: $VERSION"
}

softusb_configinit(){
	rm -f /tmp/config.tmp
	if [ -f "$DIR/config.ub" ]; then
		dd if="$DIR/config.ub" of=/tmp/config.tmp bs=72 skip=1 >/dev/null 2>&1
	else
		cp "$DIR/files/defconfig" /tmp/config.tmp	
	fi
}

softusb_configset(){
	#remove old value
	local file="`cat /tmp/config.tmp`"
	echo "$file" | grep -v -i "setenv $1 " > /tmp/config.tmp
	#add new value
	local file="`cat /tmp/config.tmp`"
	echo -e "$file\nsetenv $1 '$2'" > /tmp/config.tmp
	#remove blank lines
	local file="`cat /tmp/config.tmp`"
	echo "$file" | grep -v ^$ > /tmp/config.tmp
}

softusb_configcommit(){
	"$DIR/files/mkimage" -A sh -T script -C none -n "SoftUSB2 Config" -d /tmp/config.tmp "$DIR/config.ub" >/dev/null	
}

softusb_mount(){
#syntax softusb_mount(dev,dir)
	local DEV="`\"$DIR/files/busybox\" findfs $1`"
	mkdir -p "$2"
	[ -f /etc/mtab ] || ln -s /proc/mounts /etc/mtab
	"$DIR/files/busybox" mount -r $DEV "$2" 2>/dev/null
	return $?
}

softusb_umount(){
#syntax softusb_umount(dev,dir)
	"$DIR/files/busybox" umount "$2" 2>/dev/null
	rmdir "$2" 2>/dev/null
	return $?
}

softusb_detect(){
	[ -f "$1/usr/local/bin/enigma2" ] && DETECT=e2
	[ -f "$1/usr/bin/sbox" ] && DETECT=dgs
	if [ -f "$1/app/program/checkbootparam" ]; then
		CRC32="`\"$DIR/files/busybox\" crc32 \"$1/app/program/checkbootparam\" | cut -d\  -f1`"
		case $CRC32 in
		504392027) DETECT=pvr104;;
		82900272) DETECT=pvr105;;
		3446710112) DETECT=pvr106;;
		3205506864) DETECT=pvr107;;
		esac
	fi
	if [ -f "$1/app/app/checkbootparam" ]; then
		CRC32="`\"$DIR/files/busybox\" crc32 \"$1/app/app/checkbootparam\" | cut -d\  -f1`"
		case $CRC32 in
		4284041417) DETECT=pvr200;;
		esac
	fi
}

softusb_install(){
#syntax softusb_install(box/uboot,...)
	case $1 in
	ipbox)
		BOXTYPE=ipbox
		log info "Configuring uboot"
		softusb_checkuboot
		CHECKUBOOT=$?
		softusb_envuboot
		
		#default config 
		"$SETENV" fs 'fat' >/dev/null 2>&1
		"$SETENV" label 'IPBOX' >/dev/null 2>&1
		"$SETENV" ubootdev '0:1' >/dev/null 2>&1
		"$SETENV" dev 'usb' >/dev/null 2>&1
		
		local IPADDR="`\"$PRINTENV\" ipaddr 2>/dev/null | cut -d= -f2-`"
		if [ -z "$IPADDR" ]; then
			log warning 'Please run ./softusb2.sh set net and configure your network'
		fi
		
		"$SETENV" softusb2 '${fs}load $dev $ubootdev A6200000 /softusb2/softusb2.ub' >/dev/null 2>&1
		if [ "$2" == "2reset" ]; then
			"$SETENV" softusb2config '$dev reset; $dev reset; ${fs}load $dev $ubootdev A5FF0000 /softusb2/config.ub; if autoscr A5FF0000; then echo "SoftUSB2 config loaded"; else run menu_flash; fi' >/dev/null 2>&1
		else
			"$SETENV" softusb2config '$dev reset; ${fs}load $dev $ubootdev A5FF0000 /softusb2/config.ub; if autoscr A5FF0000; then echo "SoftUSB2 config loaded"; else run menu_flash; fi' >/dev/null 2>&1
		fi
		
		local MENU_TIMEOUT="`\"$PRINTENV\" menu_timeout_org 2>/dev/null | cut -d= -f2-`"
		if [ -z "$MENU_TIMEOUT" ]; then
			local MENU_TIMEOUT="`\"$PRINTENV\" menu_timeout 2>/dev/null | cut -d= -f2-`"
			"$SETENV" menu_timeout_org "$MENU_TIMEOUT" >/dev/null 2>&1
		fi
		"$SETENV" menu_timeout 'echo booting...; run menu_$bootmenu_defvalue; run menu_flash' >/dev/null 2>&1
		
		for i in 1 2 3 4 5 6 7 8 9; do
			local MENU_ITEM="`\"$PRINTENV\" menu_${i}_org 2>/dev/null | cut -d= -f2-`"
			if [ -z "$MENU_ITEM" ]; then
				local MENU_ITEM="`\"$PRINTENV\" menu_${i} 2>/dev/null | cut -d= -f2-`"
				"$SETENV" "menu_${i}_org" "$MENU_ITEM" >/dev/null 2>&1
			fi
			"$SETENV" "menu_$i" "run softusb2config; front_puts \"\$customname$i\"; run customboot$i; front_puts \"Boot FAILED\"; sleep 3; reset;" >/dev/null 2>&1
		done
		
		local BOOTCMD="`\"$PRINTENV\" bootcmd_org 2>/dev/null | cut -d= -f2-`"
		if [ -z "$BOOTCMD" ]; then
			local BOOTCMD="`\"$PRINTENV\" bootcmd 2>/dev/null | cut -d= -f2-`"
			"$SETENV" bootcmd_org "$BOOTCMD" >/dev/null 2>&1
		fi
		local BOOTARGS="`\"$PRINTENV\" bootargs_org 2>/dev/null | cut -d= -f2-`"
		if [ -z "$BOOTARGS" ]; then
			local BOOTARGS="`\"$PRINTENV\" bootargs 2>/dev/null | cut -d= -f2-`"
			"$SETENV" bootargs_org "$BOOTARGS" >/dev/null 2>&1
		fi
		
		"$SETENV" customboot "run bootargs0; $BOOTCMD" >/dev/null 2>&1
		"$SETENV" bootargs0 "setenv bootargs '$BOOTARGS'" >/dev/null 2>&1
		"$SETENV" menu_flash 'front_puts "Flash"; run customboot; front_puts "Boot FAILED"; sleep 3; reset' >/dev/null 2>&1
		

		touch "$DIR/softusb2.conf"
		local config="`grep -v -i BOXTYPE \"$DIR/softusb2.conf\"`"
		echo -e "BOXTYPE=ipbox\n$config" | grep -v '^$' > "$DIR/softusb2.conf"
		log info "SoftUSB configured to use IPBox uboot with bootmenu"
		;;
	1|1st|ufs910)
		BOXTYPE=ufs910
		UBOOT=1
		softusb_checkuboot
		CHECKUBOOT=$?
		softusb_envuboot

		log warning "To install 1st stage uboot refer to help. This can't be done here :("
		touch "$DIR/softusb2.conf"
		local config="`grep -v -i UBOOT \"$DIR/softusb2.conf\" | grep -v -i BOXTYPE`"
		echo -e "BOXTYPE=ufs910\nUBOOT=1\n$config" | grep -v '^$' > "$DIR/softusb2.conf"
		log info "SoftUSB configured to use 1st stage uboot"
		;;
	2|2nd|ufs910-2nd)
		BOXTYPE=ufs910
		UBOOT=2
		log info "Installing 2nd stage uboot"
		softusb_checkuboot
		CHECKUBOOT=$?
		if [ $? ]; then
			softusb_envuboot
			if [ ! -f "$DIR/files/u-boot2nd.bin" ]; then
				log error "Could not find files/u-boot2nd.bin file !!!"
				exit 1
			fi
			if [ ! -f "$DIR/files/u-boot2nd.config" ]; then
				log error "Could not find files/u-boot2nd.config file !!!"
				exit 1
			fi
			log info "Flashing 2nd stage uboot..."
			if dd if="$DIR/files/u-boot2nd.bin" of=/dev/$UBOOT2_MTDBLOCK; then
				log info "2nd stage uboot flashed successfully"
			else
				log error "2nd stage uboot flashing failed !"
				exit 1
			fi
			sync
			SUM="`\"$DIR/files/busybox\" crc32 \"$DIR/files/u-boot2nd.bin\"`"
			CRC32_DEC=`echo "$SUM" | cut -d\  -f1`
			SIZE_DEC=`echo "$SUM" | cut -d\  -f2`
			CRC32="`\"$DIR/files/busybox\" printf %x $CRC32_DEC`"
			SIZE="`\"$DIR/files/busybox\" printf %x $SIZE_DEC`"

			if [ -z "$CRC32" ]; then
				log error "Error calculating crc !\n$CRC32_DEC\n$SIZE\n$CRC32"
				exit 1
			fi

			log info "Flashing 2nd stage uboot configuration..."
			if dd if="$DIR/files/u-boot2nd.config" of=/dev/$UBOOT2CONF_MTDBLOCK; then
				log info "2nd stage uboot configuration flashed successfully"
			else
				log error "2nd stage uboot configuration flashing failed !"
				exit 1
			fi
			sync

			echo "/dev/$UBOOT1CONF_MTD 0x00000000 0x00010000 0x20000" > /tmp/su_env.config
			#backup original bootcmd
			local BOOTCMD="`\"$PRINTENV\" bootcmd_org 2>/dev/null | cut -d= -f2-`"
			local BOOTARGS="`\"$PRINTENV\" bootargs 2>/dev/null | cut -d= -f2-`"
			if [ -z "$BOOTCMD" ]; then
				local BOOTCMD="`\"$PRINTENV\" bootcmd 2>/dev/null | cut -d= -f2-`"
				"$SETENV" bootcmd_org "$BOOTCMD" >/dev/null 2>&1
			fi
			"$SETENV" bootcmd "crc32 A0FD0000 $SIZE A5000000; mw A5200000 $CRC32; if cmp A5000000 A5200000 1; then go A0FD0000; fi; run bootcmd_org; bootm A0040000" >/dev/null 2>&1
			#timeout set to 1sec
			"$SETENV" bootdelay '1' >/dev/null 2>&1

			#local ETHADDR="`\"$PRINTENV\" ethaddr 2>/dev/null | cut -d= -f2-`"
			local ETHADDR="`ifconfig eth0 | grep HWaddr | cut -dW -f2- | cut -c 6-`" # we take an address from current ifconfig 
      local IPADDR="`\"$PRINTENV\" ipaddr 2>/dev/null | cut -d= -f2-`"
			local NETMASK="`\"$PRINTENV\" netmask 2>/dev/null | cut -d= -f2-`"
			local GATEWAY="`\"$PRINTENV\" gateway 2>/dev/null | cut -d= -f2-`"
			local SERVER="`\"$PRINTENV\" serverip 2>/dev/null | cut -d= -f2-`"

			[ -z "$NETMASK" ] && local NETMASK='255.255.255.0'
			[ -z "$SERVER" ] && local SERVER="$GATEWAY"

			echo "/dev/$UBOOT2CONF_MTD 0x00000000 0x00010000 0x10000" > /tmp/su_env.config
			"$SETENV" ethaddr "$ETHADDR" >/dev/null 2>&1
			if [ -z "$IPADDR" ]; then
				log warning 'Please run ./softusb2.sh set net and configure your network'
			else
				"$SETENV" ipaddr "$IPADDR" >/dev/null 2>&1
				"$SETENV" netmask "$NETMASK" >/dev/null 2>&1
				"$SETENV" gateway "$GATEWAY" >/dev/null 2>&1
				"$SETENV" serverip "$SERVER" >/dev/null 2>&1
			fi
			"$SETENV" customboot "run bootargs0; $BOOTCMD" >/dev/null 2>&1
			"$SETENV" bootargs0 "setenv bootargs '$BOOTARGS'" >/dev/null 2>&1

			"$SETENV" softusb2 '${fs}load $dev $ubootdev A5200000 /softusb2/softusb2.ub' >/dev/null 2>&1

			touch "$DIR/softusb2.conf"
			local config="`grep -v -i UBOOT \"$DIR/softusb2.conf\" | grep -v -i BOXTYPE`"
			echo -e "BOXTYPE=ufs910\nUBOOT=2\n$config" | grep -v ^$ > "$DIR/softusb2.conf"
			log info 'SoftUSB configured to use 2nd stage uboot'
		else
			log error "Your kernel does not support 2nd stage u-boot !!!\nRefer help for information how to solve this problem"
			exit 1
		fi	
		;;
	*)
		softusb_usage
		;;
	esac
}

softusb_uninstall(){
	case $BOXTYPE in
	ipbox)
		log info "Uninstalling SoftUSB2..."
		
		local MENU_TIMEOUT="`\"$PRINTENV\" menu_timeout_org 2>/dev/null | cut -d= -f2-`"
		if [ ! -z "$MENU_TIMEOUT" ]; then
			"$SETENV" menu_timeout "$MENU_TIMEOUT" >/dev/null 2>&1
			"$SETENV" menu_timeout_org >/dev/null 2>&1
		fi
		
		for i in 1 2 3 4 5 6 7 8 9; do
			local MENU_ITEM="`\"$PRINTENV\" menu_${i}_org 2>/dev/null | cut -d= -f2-`"
			if [ ! -z "$MENU_ITEM" ]; then
				"$SETENV" "menu_$i" "$MENU_ITEM" >/dev/null 2>&1
				"$SETENV" "menu_${i}_org" >/dev/null 2>&1
			fi
		done
		
		local BOOTCMD="`\"$PRINTENV\" bootcmd_org 2>/dev/null | cut -d= -f2-`"
		if [ ! -z "$BOOTCMD" ]; then
			"$SETENV" "bootcmd" "$BOOTCMD" >/dev/null 2>&1
			"$SETENV" "bootcmd" "$BOOTCMD" >/dev/null 2>&1
		fi
		local BOOTARGS="`\"$PRINTENV\" bootargs_org 2>/dev/null | cut -d= -f2-`"
		if [ ! -z "$BOOTCMD" ]; then
			"$SETENV" "bootargs" "$BOOTCMD" >/dev/null 2>&1
			"$SETENV" "bootargs" "$BOOTCMD" >/dev/null 2>&1
		fi
		rm "$DIR/softusb2.conf" >/dev/null 2>&1
		log info "SoftUSB2 uninstalled"
		;;
	ufs910)
		case $UBOOT in
		1|1st)
			log warning "To uninstall 1st stage uboot refer to help. This can't be done here :("
			rm "$DIR/softusb2.conf" >/dev/null 2>&1
			;;
		2|2nd)
			log info "Uninstalling SoftUSB2 and 2nd stage uboot..."
			echo "/dev/$UBOOT1CONF_MTD 0x00000000 0x00010000 0x20000" > /tmp/su_env.config
			local BOOTCMD="`\"$PRINTENV\" bootcmd_org 2>/dev/null | cut -d= -f2-`"
			if [ -z "$BOOTCMD" ]; then
				"$SETENV" bootcmd 'bootm A0040000' >/dev/null 2>&1
			else
				"$SETENV" bootcmd "$BOOTCMD" >/dev/null 2>&1
				"$SETENV" bootcmd_org >/dev/null 2>&1
			fi
			rm "$DIR/softusb2.conf" >/dev/null 2>&1
			log info "SoftUSB2 uninstalled"
			;;
		*)
			softusb_usage
			;;
		esac
		;;
	*)
		log warning "Unknown or undefined boxtype. Check softusb2.conf file for details"
		;;
	esac
}

softusb_set() {
	case $1 in
	bootmenu) softusb_setbootmenu "$2" "$3"; exit 0;;
	image)	softusb_setimage "$2" "$3" "$4" "$5" "$6" "$7" "$8"; exit 0;;
	esac
	
	if [ $CHECKUBOOT -eq 0 ]; then
		case $1 in
		flash)	softusb_setflash "$2" "$3"; exit 0;;
		device)	softusb_setdevice "$2" "$3" "$4" "$5"; exit 0;;
		env)	softusb_setenv "$2" "$3"; exit 0;;
		net)	softusb_setnet "$2" "$3" "$4" "$5"; exit 0;;
		mac)	softusb_setmac "$2"; exit 0;;
		lastboot)	softusb_setlastboot "$2" "$3"; exit 0;;
		esac
	else
		log error "SoftUSB uboot not selected or selection not supported"
		exit 1
	fi

	log error "Unknown command $1"
	exit 1
}

softusb_setdevice() {
	case $1 in
	usb)	local DEV=usb;;
	ide)	local DEV=ide;;
	*)	log error "Unknown device $1"; exit 1;;
	esac
	case $2 in
	fat|vfat)	local FS=fat;;
	ext|ext2|ext3)	local FS=ext2;;
	*)		log error "Unknown filesystem $2"; exit 1;;
	esac
	if [ -z $3 ]; then
		case $BOXTYPE in
		ipbox)	local LABEL=IPBOX;local UBOOTDEV=0:1;;
		ufs910)	local LABEL=UFS910;local UBOOTDEV=UFS910;;
		esac
	else
		local LABEL=$3
		local UBOOTDEV=$4
		[ -z $4 ] && local UBOOTDEV=$LABEL
	fi

	"$SETENV" dev $DEV >/dev/null 2>&1
	"$SETENV" fs $FS >/dev/null 2>&1
	"$SETENV" label $LABEL >/dev/null 2>&1
	"$SETENV" ubootdev $UBOOTDEV >/dev/null 2>&1
	log info "SoftUSB2 configured to use $DEV device with $FS filesystem, labeled $LABEL (ubootdev $UBOOTDEV)"
}

softusb_setbootmenu() {
	case $1 in
	[1-9]);;
	*) log error "Bad argument '$1'"; exit 1;;
	esac
	case $2 in
	[1-9]|[1-4][0-9]|50);;
	*) log error "Bad argument '$2'"; exit 1;;
	esac

	softusb_configinit
	softusb_configset "bootmenucmd" "bootmenu $1 $2"
	softusb_configcommit
	log info "SoftUSB2 bootmenu set to $1 positions with timeout $2 seconds"
}

softusb_setimage() {
	case $1 in
	[1-9])	local NUM="$1";;
	*)	log error "Bad image number $1"; exit 1;;
	esac
	local TEXT="$2"
	case $4 in
	softusb*)	local LOADTYPE=softusb
			local FILE="$5"
			;;
	part)		local LOADTYPE=part
			local LABEL="$5"
			local UBOOTDEV="$6"
			[ -z "$UBOOTDEV" ] && local UBOOTDEV="$LABEL"
			;;
	partdir)	local LOADTYPE=partdir
			local OSDIR="$5"
			local LABEL="$6"
			local UBOOTDEV="$7"
			[ -z "$UBOOTDEV" ] && local UBOOTDEV="$LABEL"
			;;
	flash*)		local LOADTYPE=flash
			local LABEL="$5"
			[ -z "$LABEL" ] && local LABEL="/dev/mtdblock3"
			;;
	nfs)		local LOADTYPE=nfs
			local NFSPATH="$5"
			;;
	*)	log error "Bad loader $4"; exit 1;;
	esac
	
	if [ "$3" == "auto" ]; then
		log info "Detecting software type..."
		local TMPDIR="`\"$DIR/files/busybox\" mktemp -d -t`"
		DETECT=""
		case $LOADTYPE in
		softusb)
			if [ -f "$DIR/$FILE.part" ]; then
				local LOOPDEV="`\"$DIR/files/busybox\" losetup -f`"
				if [ ! -z "$LOOPDEV" ]; then
					"$DIR/files/busybox" losetup "$LOOPDEV" "$DIR/$FILE.part"
					if softusb_mount "$LOOPDEV" "$TMPDIR"; then
						softusb_detect "$TMPDIR"
					fi
					softusb_umount "$LOOPDEV" "$TMPDIR"
					"$DIR/files/busybox" losetup -d "$LOOPDEV" 2>/dev/null
				fi
			fi
			;;
		part*)
			if [ "`echo $LABEL | cut -c1-5`" == "/dev/" ]; then
				local tmp_arg="$LABEL"
			else
				local tmp_arg="LABEL=$LABEL"
			fi
			if softusb_mount "$tmp_arg" "$TMPDIR"; then
				softusb_detect "$TMPDIR/$OSDIR"
			fi
			softusb_umount "$tmp_arg" "$TMPDIR"
			;;
		flash)
			if softusb_mount "$LABEL" "$TMPDIR"; then
				[ "$LABEL" == "/dev/mtdblock3" ] && [ -d "$TMPDIR/app" ] && "$DIR/files/busybox" mount -r /dev/mtdblock4 "$TMPDIR/app" 2>/dev/null
				softusb_detect "$TMPDIR"
			fi
			umount "$TMPDIR/app" 2>/dev/null
			softusb_umount 
			;;
		esac
		if [ -z "$DETECT" ]; then
			log error "Could not determine software type"; exit 1
		fi
		log info "Found: $DETECT"
	else
		DETECT=$3
	fi
	
	case $BOXTYPE in
	ipbox)
		case $DETECT in
		e2|enigma|enigma2) local IMGTYPE=e2_128;;
		e2_64)	local IMGTYPE=e2_64;;
		e2_128)	local IMGTYPE=e2_128;;
		dgs)	local IMGTYPE=dgs;;
		*)	log error "Bad image type $DETECT"; exit 1;;
		esac
		local LOADADDR=A5FF0000
		local SOFTUSBADDR=A6200000
		;;
	ufs910)
		case $DETECT in
		e2|enigma|enigma2) local IMGTYPE=e2_64; local LOADADDR=A5000000;;
		e2_64)		local IMGTYPE=e2_64; local LOADADDR=A5000000;;
		e2_128)		local IMGTYPE=e2_128; local LOADADDR=A5000000;;
		104|pvr104)	local IMGTYPE=pvr104; local LOADADDR=A4000000;;
		105|pvr105)	local IMGTYPE=pvr105; local LOADADDR=A4000000;;
		106|pvr106)	local IMGTYPE=pvr106; local LOADADDR=A4000000;;
		107|pvr107)	local IMGTYPE=pvr107; local LOADADDR=A4000000;;
		200|pvr200)	local IMGTYPE=pvr200; local LOADADDR=A4000000;;
		202|pvr202)	local IMGTYPE=pvr202; local LOADADDR=A4000000;;
		*)	log error "Bad image type $DETECT"; exit 1;;
		esac
		local SOFTUSBADDR=A5200000
		;;
	*)	log error "Unknown box type"; exit 1;;
	esac

	softusb_configinit	
	if [ $LOADTYPE == softusb ]; then
		softusb_configset "bootargs$NUM" "setenv bootargs \$bootargs root=LABEL=\$label rw softusb=$FILE"
		softusb_configset "kernel$NUM" "\${fs}load \$dev \$ubootdev $LOADADDR /softusb2/$FILE.uImage; run softusb2"
		[ -f "$DIR/$FILE.part" ] || log warning "$FILE image does not exist (yet?)"
	elif [ $LOADTYPE == nfs ]; then
		softusb_configset "bootargs$NUM" "setenv bootargs \$bootargs root=/dev/nfs rw nfsroot=$NFSPATH init=/bin/devinit"
		softusb_configset "kernel$NUM" "nfs $LOADADDR \$serverip:$NFSPATH/boot/uImage"
		SOFTUSBADDR=""
	else
		if [ "`echo $LABEL | cut -c1-5`" == "/dev/" ]; then
			local tmp_arg="$LABEL"
		else
			local tmp_arg="LABEL=$LABEL"
		fi
		[ -z "$OSDIR" ] || local tmp_arg="$tmp_arg dir=$OSDIR"
		softusb_configset "bootargs$NUM" "setenv bootargs \$bootargs root=$tmp_arg rw"
		if [ $LOADTYPE == flash ]; then
			if [ $4 == flash-nosu ]; then
				softusb_configset "kernel$NUM" "echo \"no softusb\""
				SOFTUSBADDR=""
			else
				softusb_configset "kernel$NUM" "run softusb2"
			fi
			case $BOXTYPE in
			ipbox)	local LOADADDR=A0060000;;
			ufs910)	local LOADADDR=A0040000;;
			esac
		else
			softusb_configset "kernel$NUM" "ext2load \$dev $UBOOTDEV $LOADADDR $OSDIR/boot/uImage; run softusb2"
		fi
	fi

	softusb_configset "customboot$NUM" "if test \$bootmenu_defvalue != $NUM && test \$bootmenu_savedef != 0; then setenv bootmenu_defvalue $NUM; saveenv; fi; run kernel$NUM; run bootargs$BOXTYPE; run bootargs$NUM; run bootargsnet; run bootargs$IMGTYPE; bootm $LOADADDR $SOFTUSBADDR"
	softusb_configset "customname$NUM" "$TEXT"
	softusb_configcommit
	
	log info "SoftUSB2 image$NUM '$TEXT' configured to start from $LOADTYPE, software $IMGTYPE"
}

softusb_setflash() {
	local LABEL="$2"
	[ -z "$LABEL" ] && local LABEL=/dev/mtdblock3
	if [ "$1" == "auto" ]; then
		log info "Detecting software type..."
		local TMPDIR="`\"$DIR/files/busybox\" mktemp -d -t`"
		DETECT=""
		if softusb_mount "$LABEL" "$TMPDIR"; then
			[ "$LABEL" == "/dev/mtdblock3" ] && [ -d "$TMPDIR/app" ] && "$DIR/files/busybox" mount -r /dev/mtdblock4 "$TMPDIR/app" 2>/dev/null
			softusb_detect "$TMPDIR"
		fi
		umount "$TMPDIR/app" 2>/dev/null
		softusb_umount 
		if [ -z "$DETECT" ]; then
			log error "Could not determine software type"; exit 1
		fi
		log info "Found: $DETECT"
	else
		DETECT=$1
	fi
	
	case $BOXTYPE in
	ipbox)
		case $DETECT in
		e2|enigma2) local IMGTYPE=e2_128;;
		e2_64)	local IMGTYPE=e2_64;;
		e2_128)	local IMGTYPE=e2_128;;
		dgs)	local IMGTYPE=dgs;;
		*)	log error "Bad image type $DETECT"; exit 1;;
		esac
		local LOADADDR=A0060000
		;;
	ufs910)
		case $DETECT in
		e2|enigma2) local IMGTYPE=e2_64;;
		e2_64)		local IMGTYPE=e2_64;;
		e2_128)		local IMGTYPE=e2_128;;
		104|pvr104)	local IMGTYPE=pvr104;;
		105|pvr105)	local IMGTYPE=pvr105;;
		106|pvr106)	local IMGTYPE=pvr106;;
		107|pvr107)	local IMGTYPE=pvr107;;
		200|pvr200)	local IMGTYPE=pvr200;;
		202|pvr202)	local IMGTYPE=pvr202
                "$SETENV" bootargspvr202 'set bootargs $bootargs  mem=41m bigphysarea=2728 coprocessor_mem=2m@0x04000000,2m@0x04200000';;
		*)	log error "Bad image type $DETECT"; exit 1;;
		esac
		local LOADADDR=A0040000
		;;
	*)	log error "Unknown box type"; exit 1;;
	esac
	
	"$SETENV" 'customboot' "run bootargs$BOXTYPE; run bootargs$IMGTYPE; run bootargsnet; run bootargs0; bootm $LOADADDR" >/dev/null 2>&1
	"$SETENV" 'bootargs0' "setenv bootargs \$bootargs root=$LABEL" >/dev/null 2>&1	
	
	log info "SoftUSB2 flash configured to start $IMGTYPE software, from $LABEL"
}

softusb_setenv() {
	"$SETENV" $1 "$2" >/dev/null 2>&1
	log info "Uboot entry '$1' set to '$2'"
}

softusb_setlastboot() {
	case $1 in
	[1-9]);;
	*) log error "Bad argument '$1'"; exit 1;;
	esac
	case $2 in
	yes)	local SAVEDEF=1;;
	no)	local SAVEDEF=0;;
	"");;
	*) log error "Bad argument '$2'"; exit 1;;
	esac
	"$SETENV" bootmenu_defvalue "$1" >/dev/null 2>&1
	if [ ! -z "$2" ]; then
		"$SETENV" bootmenu_savedef "$SAVEDEF" >/dev/null 2>&1
		log info "Default menu item set to: $1, save last selection set to: $2"
	else
		log info "Default menu item set to: $1"
	fi
}

softusb_setnet() {
	local IPADDR="$1"
	local NETMASK="$2"
	local GATEWAY="$3"
	local SERVER="$4"
	[ -z "$SERVER" ] && local SERVER="$3"
	"$SETENV" ipaddr "$IPADDR" >/dev/null 2>&1
	"$SETENV" netmask "$NETMASK" >/dev/null 2>&1
	"$SETENV" gateway "$GATEWAY" >/dev/null 2>&1
	"$SETENV" serverip "$SERVER" >/dev/null 2>&1
	log info "Uboot (${UBOOT}) network configured\nIP address: $IPADDR\nNetmask: $NETMASK\nGateway: $GATEWAY\nServer: $SERVER"
}

softusb_setmac() {
	local MAC="$1"
	"$SETENV" ethaddr "$MAC" >/dev/null 2>&1
	log info "Uboot (${UBOOT}) network mac set to: $MAC"
}

softusb_mkimage() {
	local TARFILE="$1"
	if [ ! -f "$TARFILE" ]; then
		log error "Can't find file $TARFILE"; exit 1
	fi
	local PARTNAME="$2"
	if [ -f "$DIR/$PARTNAME.part" ]; then
		log error "File $DIR/$PARTNAME.part already exists\nDelete it manually and retry"; exit 1
	fi
	local SIZE=$3
	if [ -z $SIZE ]; then
		local SIZE=128
	fi
	if [ ! $SIZE -eq $SIZE 2>/dev/null ]; then
		log error "Bad argument $SIZE"; exit 1
	fi
	local SIZE=$(($SIZE*256))
	#sys prepare
	[ -f /etc/mtab ] || ln -s /proc/mounts /etc/mtab
	
	log info "Creating partition file..."
	if ! dd if=/dev/zero bs=4k count=$SIZE 2>/dev/null | "$DIR/files/busybox" pipe_progress | dd of="$DIR/$PARTNAME.part" 2>/dev/null; then
		log error "Error creating file"; exit 1
	fi
	sync
	local LOOPDEV="`\"$DIR/files/busybox\" losetup -f`"
	if [ -z "$LOOPDEV" ]; then
		log error "No free loop devices"; exit 1
	fi
	"$DIR/files/busybox" losetup "$LOOPDEV" "$DIR/$PARTNAME.part"
	log info "Formating partition file..."
	if ! "$DIR/files/busybox" mke2fs -L "$PARTNAME" -j -q "$LOOPDEV"; then
		"$DIR/files/busybox" losetup -d "$LOOPDEV"
		log error "Error formating file"; exit 1
	fi
	sync
	log info "Mounting partition file..."
	local TMPDIR="`\"$DIR/files/busybox\" mktemp -d -t`"
	mkdir -p "$TMPDIR"
	if ! "$DIR/files/busybox" mount "$LOOPDEV" "$TMPDIR"; then
		"$DIR/files/busybox" losetup -d "$LOOPDEV"
		log error "Error mounting image"; exit 1 
	fi
	sync
	log info "Extracting image..."
	if ! cat "$TARFILE" | "$DIR/files/busybox" pipe_progress | "$DIR/files/busybox" tar -xzf - -C "$TMPDIR" 2>/dev/null; then
		"$DIR/files/busybox" umount "$TMPDIR"
		"$DIR/files/busybox" losetup -d "$LOOPDEV"
		log error "Error extracting image"; exit 1
	fi
	if [ -d "$TMPDIR/dev_org" ]; then
		mkdir -p "$TMPDIR/dev"
		mkdir -p "$TMPDIR/mnt"
		mkdir -p "$TMPDIR/proc"
		mkdir -p "$TMPDIR/ramdisk"
		mkdir -p "$TMPDIR/var"
		mkdir -p "$TMPDIR/tmp"
	fi
	sync
	if [ -f "$TMPDIR/boot/uImage" ]; then
		mv "$TMPDIR/boot/uImage" "$DIR/$PARTNAME.uImage" >/dev/null 2>&1
	else
		log warning "Could not find uImage in archive"
	fi
	"$DIR/files/busybox" umount "$TMPDIR"
	"$DIR/files/busybox" losetup -d "$LOOPDEV"
	rmdir "$TMPDIR"
	log info "Image $PARTNAME created successfully!"
}

softusb_extract() {
#syntax softusb_extract(tar,dev,[dir]);
	local TARFILE="$1"
	if [ ! -f "$TARFILE" ]; then
		log error "Can't find file $TARFILE"; exit 1
	fi
	local LABEL="$2"
	if [ "`echo $LABEL | cut -c1-5`" == "/dev/" ]; then
		local DEVNAME="`\"$DIR/files/busybox\" findfs $LABEL`"
	else
		local DEVNAME="`\"$DIR/files/busybox\" findfs LABEL=$LABEL`"
	fi
	if [ -z "$DEVNAME" ]; then
		log error "Could not found $LABEL"; exit 1 
	fi
	local OSDIR="$3"
	[ -f /etc/mtab ] || ln -s /proc/mounts /etc/mtab
	log info "Mounting partition..."
	local TMPDIR="`\"$DIR/files/busybox\" mktemp -d -t`"
	mkdir -p "$TMPDIR"
	if ! "$DIR/files/busybox" mount "$DEVNAME" "$TMPDIR"; then
		log error "Error mounting partition"; exit 1 
	fi
	sync
	[ -d "$TMPDIR/$OSDIR" ] || mkdir -p "$TMPDIR/$OSDIR"
	log info "Extracting image..."
	if ! cat "$TARFILE" | "$DIR/files/busybox" pipe_progress | "$DIR/files/busybox" tar -xzf - -C "$TMPDIR/$OSDIR" 2>/dev/null; then
		"$DIR/files/busybox" umount "$TMPDIR"
		log error "Error extracting image"; exit 1
	fi
	if [ -d "$TMPDIR/$OSDIR/dev_org" ]; then
		mkdir -p "$TMPDIR/$OSDIR/dev"
		mkdir -p "$TMPDIR/$OSDIR/mnt"
		mkdir -p "$TMPDIR/$OSDIR/proc"
		mkdir -p "$TMPDIR/$OSDIR/ramdisk"
		mkdir -p "$TMPDIR/$OSDIR/var"
		mkdir -p "$TMPDIR/$OSDIR/tmp"
	fi
	sync
	[ -f "$TMPDIR/$OSDIR/boot/uImage" ] || log warning "Could'nt find uImage in archive"
	"$DIR/files/busybox" umount "$TMPDIR"
	rmdir "$TMPDIR"
	log info "Archive extracted successfully!"
}

softusb_mkswap() {
	local SIZE=$1
	if [ -z $SIZE ]; then
		local SIZE=64
	fi
	if [ ! $SIZE -eq $SIZE 2>/dev/null ]; then
		log error "Bad argument $SIZE"; exit 1
	fi
	local SIZE=$(($SIZE*256))
	
	if [ -f "$DIR/swap.part" ]; then
		log error "File swap.part already exists\nDelete it manually and retry"; exit 1
	fi
	
	log info "Creating swap file..."
	if ! dd if=/dev/zero of="$DIR/swap.part" bs=4k count=$SIZE; then
		log error "Error creating file"; exit 1
	fi
	sync
	log info "Formating swap file..."
	if ! "$DIR/files/busybox" mkswap "$DIR/swap.part"; then
		log error "Error formating file"; exit 1
	fi
	sync
	log info "Swap file created successfully!"
}

softusb_checkuboot(){
	case $BOXTYPE in
	ipbox)
		UBOOT1_MTD=mtd0
		UBOOT1_MTDBLOCK=mtdblock0
		UBOOT1CONF_MTD=`grep "config_welcome" /proc/mtd | cut -d: -f1`
		UBOOT1CONF_MTDBLOCK="mtdblock`echo "$UBOOT1CONF_MTD" | cut -b4-`"
		if [ ! -z $UBOOT1CONF_MTD ] && [ -b /dev/$UBOOT1CONF_MTDBLOCK ]; then
			return 0
		fi
		;;
	ufs910)
		UBOOT1_MTD=mtd0
		UBOOT1_MTDBLOCK=mtdblock0
		UBOOT1CONF_MTD=`grep "0xA002.0000-0xA003.FFFF" /proc/mtd | cut -d: -f1`
		UBOOT1CONF_MTDBLOCK="mtdblock`echo "$UBOOT1CONF_MTD" | cut -b4-`"
		UBOOT2_MTD=`grep "0xA0FD.0000-0xA0FF.FFFF" /proc/mtd | cut -d: -f1`
		UBOOT2_MTDBLOCK="mtdblock`echo "$UBOOT2_MTD" | cut -b4-`"
		UBOOT2CONF_MTD=`grep "0xA0FC.0000-0xA0FC.FFFF" /proc/mtd | cut -d: -f1`
		UBOOT2CONF_MTDBLOCK="mtdblock`echo "$UBOOT2CONF_MTD" | cut -b4-`"
		[ -z "$UBOOT" ] && return 1
		if [ "$UBOOT" == "1" ] && [ ! -z $UBOOT1CONF_MTD ] && [ -b /dev/$UBOOT1CONF_MTDBLOCK ]; then
			return 0
		elif [ "$UBOOT" == "2" ] && [ ! -z $UBOOT2_MTD ] && [ ! -z $UBOOT2CONF_MTD ] && [ -b /dev/$UBOOT2_MTDBLOCK ] && [ -b /dev/$UBOOT2CONF_MTDBLOCK ]; then
			return 0
		fi
		;;
	esac
	return 1
}

softusb_envuboot(){
	if [ $CHECKUBOOT -eq 0 ]; then
		case $BOXTYPE in
		ipbox)
			echo "/dev/$UBOOT1CONF_MTD 0x00000000 0x00020000 0x20000" > /tmp/su_env.config
			;;
		ufs910)
			if [ "$UBOOT" == "1" ]; then
				echo "/dev/$UBOOT1CONF_MTD 0x00000000 0x00010000 0x20000" > /tmp/su_env.config
			else
				echo "/dev/$UBOOT2CONF_MTD 0x00000000 0x00010000 0x10000" > /tmp/su_env.config
			fi
			;;
		esac
	fi	
}

softusb_status(){
if [ $CHECKUBOOT -eq 0 ]; then
#network
	local ETHADDR="`\"$PRINTENV\" ethaddr 2>/dev/null | cut -d= -f2-`"
	local IPADDR="`\"$PRINTENV\" ipaddr 2>/dev/null | cut -d= -f2-`"
	local NETMASK="`\"$PRINTENV\" netmask 2>/dev/null | cut -d= -f2-`"
	local GATEWAY="`\"$PRINTENV\" gateway 2>/dev/null | cut -d= -f2-`"
	local SERVER="`\"$PRINTENV\" serverip 2>/dev/null | cut -d= -f2-`"
	log info "--== NETWORK ==--
	MAC:\t$ETHADDR
	IP:\t$IPADDR
	MASK:\t$NETMASK
	GATEWAY:\t$GATEWAY
	SERVER:\t$SERVER"
#flash
	local CUSTOMBOOT="`\"$PRINTENV\" customboot 2>/dev/null | cut -d= -f2-`"
	local BOOTARGS0="`\"$PRINTENV\" bootargs0 2>/dev/null | cut -d= -f2-`"
	if [ ! -z "$CUSTOMBOOT" ]; then
		log info "--== FLASH CONFIG ==--
	BOOTCMD:\t$CUSTOMBOOT
	BOOTARGS:\t$BOOTARGS0"
	fi
fi
if [ -f "$DIR/config.ub" ]; then
	softusb_configinit
	#bootmenu
	local BOOTMENU="`cat /tmp/config.tmp | grep \"setenv bootmenucmd\" | cut -d\' -f2-`"
	if [ -z "$BOOTMENU" ]; then
		log info "--== BOOTMENU ==--
	Bootmenu not configured"
	else
		local NUM="`echo \"$BOOTMENU\" | cut -d\  -f 2`"
		local TIMEOUT="`echo \"$BOOTMENU\" | cut -d\  -f 3`"
		log info "--== BOOTMENU ==--
	NUMBER:\t$NUM
	TIMEOUT:\t$TIMEOUT"
	fi
	#image
	log info "--== IMAGES ==--"
	for i in 1 2 3 4 5 6 7 8 9; do
		local IMAGENAME="`cat /tmp/config.tmp | grep \"setenv customname$i\" | cut -c20-`"
		if [ ! -z "$IMAGENAME" ]; then
			log "\tIMAGE$i:\t$IMAGENAME"
		fi
	done
fi
}

#load config
DIR="`dirname \"$0\" 2>/dev/null`"
if [ -z "$DIR" ]; then
	DIR="${0%/*}"
	if [ -z "$DIR" ]; then
		DIR="`pwd`"
	fi
fi
if [ "`echo \"$DIR\" | cut -c1`" == "." ]; then
	DIR="`pwd``echo \"$DIR\" | cut -c2-`"
fi
[ -f "$DIR/softusb2.conf" ] && . "$DIR/softusb2.conf"

softusb_checkuboot
CHECKUBOOT=$?
softusb_envuboot

SETENV="$DIR/files/su_setenv"
PRINTENV="$DIR/files/su_printenv"

if [ "`\"$DIR/files/busybox\" uname -r 2>/dev/null`" == "2.6.17.14_stm22_0037" ]; then
	DEVNAME="`\"$DIR/files/busybox\" mountpoint -n \"$DIR\" | cut -d\  -f1`"
	mount -o remount,sync $DEVNAME 2>/dev/null
	insmod "$DIR/files/loop.ko" 2>/dev/null
fi

#commands
case $1 in
	install)	softusb_install "$2" "$3" "$4";;
	set)		softusb_set "$2" "$3" "$4" "$5" "$6" "$7" "$8" "$9";;
	mkimage)	softusb_mkimage "$2" "$3" "$4";;
	mkswap)		softusb_mkswap "$2";;
	extract)	softusb_extract "$2" "$3" "$4";;
	uninstall)	softusb_uninstall "$2";;
	status)		softusb_status;;
	*)		softusb_usage;;
esac

sync
